/************************************************************************************************************\

Module Name:    LBuffer.h

Description:    Describe how to allocate and use Liberatus buffer.
                Region of memory used to temporarily hold data.

References:     LBuffer.doc Revision 0.14.

    Copyright (c) 2013, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LBUFFER_H
#define INC_LBUFFER_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "Liberatus.h"
#include "LPixelFormat.h"

typedef struct LBUFFER_OBJECT* LBuffer_Handle;
// -----------------------------------------------------------------------------------------------------------
//                                   O V E R V I E W
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\
    A buffer is a region of memory used to temporarily hold data while it's being moved from one place to
    another. In Liberatus, in addition to the moving process, some transformations can be scheduled. Memory
    used by the buffer can be allocated in the local, system or external memory.
\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------
#define LBUFFER_MAXPLANE        3       // Maximum number of plane per video buffer
#define LBUFFER_MAXPRIVATEDATA  8       // Maximum number of private data in buffer.

/************************************************************************************************************\

Enum:           LBuffer_PlaneID

Description:    Specifies the Plane ID of the buffer.

Comments:       None.

\************************************************************************************************************/
typedef enum {
    LBuffer_PlaneID_0                          = 0,          //  Plane 0
    LBuffer_PlaneID_1                          = 1,          //  Plane 1
    LBuffer_PlaneID_2                          = 2,          //  Plane 2
    LBuffer_PlaneID_ALL                        = 0x7FFFFFFF, //  All the Planes
} LBuffer_PlaneID;

/************************************************************************************************************\

Enum:           LBuffer_Type

Description:    Specifies the type of the buffer.

Comments:       None.

\************************************************************************************************************/
typedef enum {
    LBuffer_Type_NONE                       = 0,       // Unspecified buffer type. This value is considered
                                                       // invalid and should never be used explicitly.
    LBuffer_Type_LINEAR                     = 1,       // Buffer without specialized attributes.
    LBuffer_Type_VIDEO                      = 2,       // Buffer with video attributes.
    LBuffer_Type_SYSTEM_LINEAR              = 3,       // System buffer without specialized attributes.
    LBuffer_Type_SYSTEM_VIDEO               = 4,       // System buffer with video attributes.
    LBuffer_Type_EXTERNAL_LINEAR            = 5,       // External linear buffer.
    LBuffer_Type_EXTERNAL_VIDEO             = 6,       // External video buffer.
    LBuffer_Type_FROM_BUS_DESCRIPTOR_LINEAR = 7,       // Used for inter-device sharing of a linear buffer.
    LBuffer_Type_FROM_BUS_DESCRIPTOR_VIDEO  = 8,       // Used for inter-device sharing of a video buffer.
    LBuffer_Type_FROM_LUID                  = 9,       // Used for inter-process sharing buffer.
    LBuffer_Type_FORCE32                    = 0x7FFFFFFF,   // Dummy value to force to use 32-bits.
} LBuffer_Type;

/************************************************************************************************************\

Enum:           LBuffer_BusType

Description:    Specifies a bus.

Comments:       None.

\************************************************************************************************************/
typedef enum {
    LBuffer_BusType_SYSTEM             = 0,             // System memory of the CPU.
    LBuffer_BusType_EXTERNAL           = 1,             // External device connected on PCIe.
    LBuffer_BusType_FORCE32            = 0x7FFFFFFF,    // Dummy value to force to use 32-bits.
} LBuffer_BusType;


/************************************************************************************************************\

Structure:      LBuffer_PlaneInfo

Description:    Holds information about plane for direct access.

Comments:
\************************************************************************************************************/
typedef struct tagLBuffer_PlaneInfo
{
    MUINT32      uiPlaneCount;                        // Number of plane in the video frame.
    LPixelFormat aePlaneFormat[LBUFFER_MAXPLANE];     // Pixel format representation.
    MUINT32      auiWidth[LBUFFER_MAXPLANE];          // Width in pixels of the video frame.
    MUINT32      auiWidthInBytes[LBUFFER_MAXPLANE];   // Width in bytes of the video frame.
    MUINT32      auiHeight[LBUFFER_MAXPLANE];         // Height in lines of the video frame.
    MUINT32      auiPitch[LBUFFER_MAXPLANE];          // Number of pixels per line.
    MUINT32      auiPitchInBytes[LBUFFER_MAXPLANE];   // Number of bytes per line.
} LBuffer_PlaneInfo;

/************************************************************************************************************\

Structure:      LBuffer_LinearAttributes

Description:    Describes linear buffer attributes.

Comments:       LBuffer_Type_LINEAR.

\************************************************************************************************************/
typedef struct tagLBuffer_LinearAttributes
{
    LBuffer_Type                eAttributeType;     // Shall be LBuffer_Type_LINEAR.
    MUINT32                     uiSize;             // Size, in bytes, of this buffer.
} LBuffer_LinearAttributes;


/************************************************************************************************************\

Structure:      LBuffer_VideoAttributes

Description:    Describes video frame attributes of a buffer.

Comments:       LBuffer_Type_VIDEO.

\************************************************************************************************************/
typedef struct tagLBuffer_VideoAttributes
{
    LBuffer_Type            eAttributeType;     // Shall be LBuffer_Type_VIDEO.
    LPixelFormat            ePixelFormat;       // Pixel format representation.
    MUINT32                 uiWidth;            // Width, in pixels, of the video frame.
    MUINT32                 uiHeight;           // Height, in lines, of the video frame.
} LBuffer_VideoAttributes;


/************************************************************************************************************\

Structure:      LBuffer_SystemLinearAttributes

Description:    Describes linear system buffer attributes.

Comments:       LBuffer_Type_SYSTEM_LINEAR.

\************************************************************************************************************/
typedef struct tagLBuffer_SystemLinearAttributes
{
    LBuffer_Type                eAttributeType;     // Shall be LBuffer_Type_SYSTEM_LINEAR.
    MUINT32                     uiSize;             // Size, in bytes, of this buffer.
    void*                       pvSystemMemory;     // Pointer to system memory. Can be Null when used with
                                                    // LBuffer_Create. In this case, memory will be allocated
                                                    // by function.
} LBuffer_SystemLinearAttributes;


/************************************************************************************************************\

Structure:      LBuffer_SystemVideoAttributes

Description:    Describes video frame attributes of a system buffer.

Comments:       LBuffer_Type_SYSTEM_VIDEO.

\************************************************************************************************************/
typedef struct tagLBuffer_SystemVideoAttributes
{
    LBuffer_Type        eAttributeType;     // Shall be LBuffer_Type_SYSTEM_VIDEO.
    LPixelFormat        ePixelFormat;       // Pixel format representation.
    MUINT32             uiWidth;            // Width, in pixels, of the video frame.
    MUINT32             uiHeight;           // Height, in lines, of the video frame.
    MUINT32             uiSize;             // Size, in bytes, of this buffer.
    void*               pvSystemMemory;     // Pointer to system memory. Can be Null when used with
                                            //  LBuffer_Create. In this case, memory will be allocated by
                                            //  function and uiSize will be updated.
} LBuffer_SystemVideoAttributes;

/************************************************************************************************************\

Structure:      LBuffer_MemoryDescriptor

Description:    Holds a memory descriptor for a given buffer.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLBuffer_MemoryDescriptor
{
    MUINT64 uiPhysicalAddress;  // Physical address described by this block.
    MUINT32 uiBlockSize;        // Size, in bytes, of this memory block.
} LBuffer_MemoryDescriptor;


/************************************************************************************************************\

Structure:      LBuffer_ExternalLinearAttributes

Description:    Describes linear external buffer attributes.

Comments:       LBuffer_Type_EXTERNAL_LINEAR.

\************************************************************************************************************/
typedef struct tagLBuffer_ExternalLinearAttributes
{
    LBuffer_Type                eAttributeType; // Shall be LBuffer_Type_EXTERNAL_LINEAR.
    MUINT32                     uiArraySize;    // Size of the descriptors array.
    LBuffer_MemoryDescriptor*   poDescriptors;  // Array of physical address to map to the buffer.
} LBuffer_ExternalLinearAttributes;


/************************************************************************************************************\

Structure:      LBuffer_ExternalVideoAttributes

Description:    Holds the buffer attributes relative to the external video frame.

Comments:       LBuffer_Type_EXTERNAL_VIDEO.

\************************************************************************************************************/
typedef struct tagLBuffer_ExternalVideoAttributes
{
    LBuffer_Type              eAttributeType;   // Shall be LBuffer_Type_EXTERNAL_VIDEO.
    LPixelFormat              ePixelFormat;     // Pixel format representation.
    MUINT32                   uiWidth;          // Width, in pixels, of the video frame.
    MUINT32                   uiHeight;         // Height, in lines, of the video frame.
    MUINT32                   uiArraySize;      // Descriptors array size.
    LBuffer_MemoryDescriptor* poDescriptors;    // Array of physical addresses to map to the buffer.
} LBuffer_ExternalVideoAttributes;

/************************************************************************************************************\

Structure:      LBuffer_BusDescriptorLinearAttributes

Description:    Holds the linear attributes of a buffer created from a bus descriptor.

Comments:       LBuffer_Type_FROM_BUS_DESCRIPTOR_LINEAR.

\************************************************************************************************************/
typedef struct tagLBuffer_BusDescriptorLinearAttributes
{
    LBuffer_Type             eAttributeType;    // Shall be LBuffer_Type_FROM_BUS_DESCRIPTOR_LINEAR.
    LBuffer_BusType          eBusType;          // Bus type associated with the memory descriptor.
    LBuffer_MemoryDescriptor oDescriptor;       // Memory descriptor to use for creating the buffer.
} LBuffer_BusDescriptorLinearAttributes;

/************************************************************************************************************\

Structure:      LBuffer_BusDescriptorVideoAttributes

Description:    Holds the video attributes of a buffer created from a bus descriptor.

Comments:       LBuffer_Type_FROM_BUS_DESCRIPTOR_VIDEO.

\************************************************************************************************************/
typedef struct tagLBuffer_BusDescriptorVideoAttributes
{
    LBuffer_Type             eAttributeType;    // Shall be LBuffer_Type_FROM_BUS_DESCRIPTOR_VIDEO.
    LPixelFormat             ePixelFormat;      // Pixel format representation.
    MUINT32                  uiWidth;           // Width, in pixels, of the video frame.
    MUINT32                  uiHeight;          // Height, in lines, of the video frame.
    LBuffer_BusType          eBusType;          // Bus type associated with the memory descriptor.
    LBuffer_MemoryDescriptor oDescriptor;       // Memory descriptor to use for creating the buffer.
} LBuffer_BusDescriptorVideoAttributes;

/************************************************************************************************************\

Structure:      LBuffer_LUID

Description:    Specifies the LUID from other processes that can be used to create buffer sharing in the same
                 local memory in local process.

Comments:       LBuffer_Type_FROM_LUID.

\************************************************************************************************************/
typedef struct tagLBuffer_LUID
{
    LBuffer_Type    eAttributeType;             // Type of buffer.
    MUINT64         uiLUID;                     // Locally unique identifier
} LBuffer_LUID;

/************************************************************************************************************\

Structure:      LBuffer_Attributes

Description:    Holds the video attributes of a any type of buffer.

Comments:       eAttributeType must be set by user.

\************************************************************************************************************/
typedef union tagLBuffer_Attributes
{
        LBuffer_Type                            eAttributeType;                 // Type of buffer.
        LBuffer_LinearAttributes                oLinearAttributes;
        LBuffer_VideoAttributes                 oVideoAttributes;
        LBuffer_SystemLinearAttributes          oSystemLinearAttributes;
        LBuffer_SystemVideoAttributes           oSystemVideoAttributes;
        LBuffer_ExternalLinearAttributes        oExternalLinearAttributes;
        LBuffer_ExternalVideoAttributes         oExternalVideoAttributes;
        LBuffer_BusDescriptorLinearAttributes   oBusDescriptorLinearAttributes;
        LBuffer_BusDescriptorVideoAttributes    oBusDescriptorVideoAttributes;
        LBuffer_LUID                            oLUID;
} LBuffer_Attributes;

/************************************************************************************************************\

Enum:           LBuffer_OptionType

Description:    Specifies the type of option.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LBuffer_OptionType_AutoSync = 0x00000001,   // Auto-synchronization option

} LBuffer_OptionType;

/************************************************************************************************************\

Enum:           LBuffer_AutoSyncOption

Description:    Specifies options for auto sync.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LBuffer_AutoSyncOption_Disabled,    // Disable auto-synchronization.
    LBuffer_AutoSyncOption_PerBuffer,   // Enable auto-synchronization on the whole buffer.
    LBuffer_AutoSyncOption_PerRegion,   // Enable auto-synchronization on sub-regions in the buffer.

} LBuffer_AutoSyncOption;

/************************************************************************************************************\

Structure:      LBuffer_Options

Description:    Structure used to set the buffer options.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLBuffer_Options
{
    MUINT32                 uiOptionTypes;      // Bits field of LBuffer_OptionType to select the available
                                                // options.
    LBuffer_AutoSyncOption  eAutoSyncOption;    // Auto-synchronization option. Valid only if
                                                // LBuffer_OptionType_AutoSync bit is set in uiOptionTypes.

    MUINT32                 uiAutoSyncTimeout;  // Timeout when doing synchronization. Valid only if
                                                // LBuffer_OptionType_AutoSync bit is set in
                                                // uiOptionTypes.
} LBuffer_Options;

/************************************************************************************************************\

Function:       LBuffer_Callback

Description:    Describes a generic callback function.

Parameters:     IN  hBuffer                 Handle of a buffer.
                IN  pvContext               Pointer to a context associated with the callback.

Return Value:   LStatus.

Comments:       None.

\************************************************************************************************************/
typedef LStatus (*LBuffer_Callback)(LBuffer_Handle hBuffer, void* pvContext);


// -----------------------------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                      U S A G E
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LBuffer_Create

Description:    Creates a buffer using local, system or external memory.

Parameters:     IN  hDev                    Handle of the device object for which to create the buffer.
                IN  peAttributeType         Pointer to the parameter type member of a structure with
                                            parameters needed to create buffer.
                OUT phBuffer                Pointer to LBuffer_Handle. Upon successful creation, this is 
                                            set to the handle of the newly created buffer. MNULL if the 
                                            creation failed.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_MEMORY       Insufficient memory.

Comments:       Creates a buffer of the predefined parameters of the type pointed to by peAttributeType.

                When creating a buffer from a bus memory descriptor (LBuffer_Type_FROM_BUS_DESCRIPTOR_LINEAR, 
                LBuffer_Type_FROM_BUS_DESCRIPTOR_VIDEO), the memory descriptor is associated with a device on 
                the system or external PCIe bus. Physical addresses that correspond to system or external 
                memory aren’t permitted.

                When creating an external memory buffer (LBuffer_Type_EXTERNAL_LINEAR, 
                LBuffer_Type_EXTERNAL_VIDEO), the memory descriptors is associated with external memory. 
                Physical addresses that correspond to a device on the external PCIe bus aren’t permitted.
\************************************************************************************************************/
LAPI LStatus LBuffer_Create(
    LDevice_Handle  hDev,
    LBuffer_Type*   peAttributeType,
    LBuffer_Handle* phBuffer);

/************************************************************************************************************\

Function:       LBuffer_Derive

Description:    Derives a buffer using local, system or external memory.

Parameters:     IN  hDev                    Handle of the device object for which to create the buffer.
                IN  hBufferIn               Pointer to LBuffer_Handle of the "source" buffer.
                IN  ePlaneID                Selected plane ID of the buffer.
                IN  eDerivedPixelFormat     Pixel format of the derived buffer.
                OUT phBufferOut             Pointer to LBuffer_Handle of the derived buffer. Upon successful derivation, this is
                                            set to the handle of the newly derived buffer. MNULL if the derivation failed.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_MEMORY       Insufficient memory.

Comments:       Derives a buffer from an other buffer with a different pixel format.

\************************************************************************************************************/
LAPI LStatus LBuffer_Derive(
    LDevice_Handle  hDev,
    LBuffer_Handle  hBufferIn,
    LBuffer_PlaneID ePlaneID,
    LPixelFormat    eDerivedPixelFormat,
    LBuffer_Handle* phBufferOut);

/************************************************************************************************************\

Function:       LBuffer_CreateLinear

Description:    Shortcut to create a buffer of type LBuffer_Type_LINEAR.

Parameters:     IN  hDev                    Handle of the device object for which to create the buffer.
                IN uiSize                   Size in bytes.
                OUT phBuffer                Pointer to LBuffer_Handle. Upon successful creation, this is set 
                                            to the handle of the newly created buffer. MNULL if the creation 
                                            failed.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_MEMORY       Insufficient memory.

Comments:       - Creates a buffer with linear attributes.

\************************************************************************************************************/
LAPI LStatus LBuffer_CreateLinear(
    LDevice_Handle  hDev,
    MUINT32         uiSize,
    LBuffer_Handle* phBuffer);

/************************************************************************************************************\

Function:       LBuffer_CreateVideo

Description:    Shortcut to create a buffer of type LBuffer_Type_VIDEO.

Parameters:     IN  hDev                    Handle of the device object for which to create the buffer.
                IN  ePixelFormat            Pixel format representation.
                IN  uiWidth                 Width, in pixels, of the video frame.
                IN  uiHeight                Height, in lines, of the video frame.
                OUT phBuffer                Pointer to LBuffer_Handle. Upon successful creation, this is set 
                                            to the handle of the newly created buffer. MNULL if the creation 
                                            failed.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_MEMORY       Insufficient memory.

Comments:       Creates a buffer with video attributes.

\************************************************************************************************************/
LAPI LStatus LBuffer_CreateVideo(
    LDevice_Handle  hDev,
    LPixelFormat    ePixelFormat,
    MUINT32         uiWidth,
    MUINT32         uiHeight,
    LBuffer_Handle* phBuffer);


/************************************************************************************************************\

Function:       LBuffer_AddRef

Description:    Increments the reference count of the specified buffer.

Parameters:     IN  hBuffer                 Handle identifying the buffer on which to perform the operation.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - Use LBuffer_AddRef to prevent the buffer from being destroyed. Resources for the buffer are
                  released only when the reference to the buffer reaches 0. At creation, the reference number
                  on the buffer is set to 1. Calling LDestroy_Buffer on a buffer that has more than one
                  reference on it won't release the resources.

\************************************************************************************************************/
LAPI LStatus LBuffer_AddRef(LBuffer_Handle  hBuffer);

/************************************************************************************************************\

Function:       LBuffer_GetRefCount

Description:    Gets the current reference count of the specified buffer.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.

Return Value:   MUINT32             Current reference count of the specified buffer.

Comments:       None.

\************************************************************************************************************/
LAPI MUINT32 LBuffer_GetRefCount(LBuffer_Handle  hBuffer);


/************************************************************************************************************\

Function:       LBuffer_UnRef

Description:    Decrements the reference count of a specific buffer.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - The resources allocated for this buffer are released only when the internal reference count
                  reaches 0.

\************************************************************************************************************/
LAPI LStatus LBuffer_UnRef(LBuffer_Handle hBuffer);


/************************************************************************************************************\

Function:       LBuffer_Destroy

Description:    Alias for LBuffer_UnRef. When creating a buffer, use LBuffer_Destroy for code clarity.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_REFCOUNT_NOTZERO    References remain on the buffer, the resource won't be released.

Comments:       Alias for LBuffer_UnRef. When creating a buffer, use LBuffer_Destroy for code clarity.
                
                Destroy won’t fail if a reference is present on the buffer. The resources are only released 
                when the last reference is removed.

\************************************************************************************************************/
LAPI LStatus LBuffer_Destroy(LBuffer_Handle hBuffer);

/************************************************************************************************************\

Function:       LBuffer_RegisterDestroyCallback

Description:    Registers a callback function on the destroy event.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.
                IN  pfnCallback     Pointer to a callback function.
                IN  pvContext       Pointer to a context associated with the callback function.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       This function registers a callback function and the associated context with the buffer. 
                When the buffer is about to be destroyed, each registered callback function is called to 
                notify the recipient about the destruction of the buffer.

                A callback is uniquely identified by the function pointer and the associated context 
                pointer. A callback can be unregistered using these two pieces of information.
                
                It’s an error to register a callback more than once. Registering a callback function more 
                than once is permitted only when the associated context pointer differs in each registration.

                The processing within the callback function is multi-thread safe, as other processor threads 
                may compete for accessing the callback context.

                The callback function can’t call any method of the buffer object.

                The callback function must assess the validity of the context pointer. The callback is 
                unregistered before the associated context pointer becomes invalid.

\************************************************************************************************************/
LAPI LStatus LBuffer_RegisterDestroyCallback(
    LBuffer_Handle   hBuffer,
    LBuffer_Callback pfnCallback,
    void*            pvContext);

/************************************************************************************************************\

Function:       LBuffer_UnregisterDestroyCallback

Description:    Unregisters a destroy callback function.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.
                IN  pfnCallback     Pointer to a callback function.
                IN  pvContext       Pointer to a context associated with the callback function.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       This function unregisters a destroy callback from the buffer.

                A callback is uniquely identified by the function pointer and the associated context
                pointer. A callback can be unregistered using these two pieces of information.

\************************************************************************************************************/
LAPI LStatus LBuffer_UnregisterDestroyCallback(
    LBuffer_Handle   hBuffer,
    LBuffer_Callback pfnCallback,
    void*            pvContext);

/************************************************************************************************************\

Function:       LBuffer_SetPrivateData

Description:    Stores a pointer to any private data to be associated with the buffer.

Parameters:     IN  hBuffer                 Handle identifying the buffer on which to perform the operation.
                IN  uiPrivataDataIndex      Private's data index.
                IN  pPrivateData            Pointer to the private data.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       LBUFFER_MAXPRIVATEDATA  is the maximum number of private data in buffer.

\************************************************************************************************************/
LAPI LStatus LBuffer_SetPrivateData(
    LBuffer_Handle hBuffer,
    MUINT32 uiPrivataDataIndex,
    MUINT64 uiPrivateData);

/************************************************************************************************************\

Function:       LBuffer_GetPrivateData

Description:    Retrieves the previously specified pointer to private data associated with the buffer.

Parameters:     IN  hBuffer                 Handle identifying the buffer on which to perform the operation.
                IN  uiPrivataDataIndex      Private data index.
                OUT puiPrivateData          Private data associated with the specified buffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       LBUFFER_MAXPRIVATEDATA  is the maximum number of private data in buffer.

\************************************************************************************************************/
LAPI LStatus LBuffer_GetPrivateData(
    LBuffer_Handle hBuffer,
    MUINT32        uiPrivataDataIndex,
    MUINT64*       puiPrivateData);



/************************************************************************************************************\

Function:       LBuffer_GetLUID

Description:    Gets the unique identifier for this buffer. It can be used to create a new buffer that shares 
                the same local memory on another process. (See the LBuffer_Type_FROM_LUID member.)

Parameters:     IN  hBuffer                 Handle identifying the buffer on which to perform the operation.
                OUT puiLUID                 Pointer to the unique ID for the buffer in this process.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       

\************************************************************************************************************/
LAPI LStatus LBuffer_GetLUID(LBuffer_Handle hBuffer, MUINT64* puiLUID);


/************************************************************************************************************\

Function:       LBuffer_BeginAccess

Description:    Maps the buffer memory into the user process address space.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.
                IN  flAccessFlags   Type of access requested.
                IN  uiArraySize     Size of the apuiData array.
                OUT apuiData        Array of bytes pointers.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_RESOURCES    Insufficient system resources. To guarantee the availability
                                            of the resource.

Comments:       - flAccessFlag is reserved for future usage.  Must set to 0.

\************************************************************************************************************/
LAPI LStatus LBuffer_BeginAccess(
    LBuffer_Handle hBuffer,
    MFLAG32        flAccessFlags,
    MUINT32        uiArraySize,
    MUINT8*        apuiData[]);

/************************************************************************************************************\

Function:       LBuffer_EndAccess

Description:    Unmaps the buffer memory from the user process address space.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.


Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_EndAccess(LBuffer_Handle hBuffer);

/************************************************************************************************************\

Function:       LBuffer_BeginBusAccess

Description:    Maps the local memory buffer into the specified bus aperture.

Parameters:     IN  hBuffer                 Handle identifying the buffer on which to perform the operation.
                IN  eBusType                Bus where to create aperture.
                IN  uiArraySize             Size of the apuiData array.
                OUT aoDescriptor            Array of physical address and size of the aperture in the
                                            specified bus.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters. Unsupported bus or
                                             invalid range.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_RESOURCES    Insufficient system resources.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_BeginBusAccess(
    LBuffer_Handle             hBuffer,
    LBuffer_BusType            eBusType,
    MUINT32                    uiArraySize,
    LBuffer_MemoryDescriptor*  aoDescriptor);

/************************************************************************************************************\

Function:       LBuffer_EndBusAccess

Description:    Unmaps the local memory buffer from the specified bus aperture.

Parameters:     IN  hBuffer                 Handle identifying the buffer on which to perform the operation.
                IN  eBusType                Bus where to unmap aperture.
                IN  poRange                 Memory range in buffer to map. Null pointer will map entire buffer.
                IN  puiSize                 Size of the aperture where we map buffer memory.
                OUT puiPhysicalAddresses    The physical address of aperture in specified bus.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_EndBusAccess(LBuffer_Handle hBuffer, LBuffer_BusType eBusType);

/************************************************************************************************************\

Function:       LBuffer_GetSize

Description:    Retrieves the number of bytes in memory occupied by the specified buffer.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.

Return Value:   MUINT32             Number of bytes in memory occupied by the specified buffer.

Comments:       None.

\************************************************************************************************************/
LAPI MUINT32 LBuffer_GetSize(LBuffer_Handle  hBuffer);


/************************************************************************************************************\

Function:       LBuffer_GetType

Description:    Retrieves the type of the specified buffer.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.

Return Value:   LBuffer_Handle      Type of the buffer.

Comments:       None.

\************************************************************************************************************/
LAPI LBuffer_Type  LBuffer_GetType(LBuffer_Handle  hBuffer);



/************************************************************************************************************\

Function:       LBuffer_GetAttributes

Description:    Retrieves the attributes of the specified buffer.

Parameters:     IN      hBuffer         Handle identifying the buffer on which to perform the operation.
                IN OUT  peAttributeType Pointer to the parameter type member of a structure to be filled with
                                         attributes for this buffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                GetAFunction failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_GetAttributes(LBuffer_Handle hBuffer, LBuffer_Type* peAttributeType);

/************************************************************************************************************\

Function:       LBuffer_GetVideoAttributes

Description:    Retrieves the video attributes of the specified buffer.

Parameters:     IN   hBuffer                Handle identifying the buffer on which to perform the operation.
                OUT  poVideoAttributes      Pointer of structure to be filled with video attributes for this
                                            buffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                GetAFunction failed.

Comments:       For linear buffer poVideoAttributes->ePixelFormat = LPixelFormat_TYPELESS8 and
                poVideoAttributes->uiHeight = 1.
\************************************************************************************************************/
LAPI LStatus LBuffer_GetVideoAttributes(LBuffer_Handle hBuffer, LBuffer_VideoAttributes* poVideoAttributes);

/************************************************************************************************************\

Function:       LBuffer_GetPlaneInfo

Description:    Retrieves information about plane for direct access.

Parameters:     IN  hBuffer         Handle identifying the buffer on which to perform the operation.
                OUT poPlaneInfo     Pointer to the parameter type member of a structure to be filled
                                     with information  for this buffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_GetPlaneInfo(LBuffer_Handle hBuffer, LBuffer_PlaneInfo*  poPlaneInfo);

/************************************************************************************************************\

Function:       LBuffer_GetMemoryDescriptors

Description:    Retrieves the memory descriptors from system or external buffer.

Parameters:     IN     hBuffer         Handle identifying the buffer on which to perform the operation.
                IN OUT puiArraySize    Pointer to the Size of the descriptors array.
                OUT    poDescriptors   Array of the physical addresses on this buffer (can be null).

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_UNSUPPORTED         No descriptors for this type buffer.
                LStatus_FAIL                Function failed.

Comments:       puiArraySize must be set  with size of poDescriptor array. If you only need the array size,
                set poDescriptor to null.

\************************************************************************************************************/
LAPI LStatus LBuffer_GetMemoryDescriptors(
    LBuffer_Handle            hBuffer,
    MUINT32*                  puiArraySize,
    LBuffer_MemoryDescriptor* poDescriptors);

/************************************************************************************************************\

Function:       LBuffer_SetOptions

Description:    Set buffer options.

Parameters:     IN  hBuffer     Handle identifying the buffer on which to set the options.
                IN  poOptions   Pointer to the options structure.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_SetOptions(
    LBuffer_Handle      hBuffer,
    LBuffer_Options*    poOptions);

/************************************************************************************************************\

Function:       LBuffer_GetOptions

Description:    Get buffer options.

Parameters:     IN  hBuffer     Handle identifying the buffer from which to get the options.
                OUT poOptions   Pointer to the options structure.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_GetOptions(
    LBuffer_Handle      hBuffer,
    LBuffer_Options*    poOptions);

/************************************************************************************************************\

Function:       LBuffer_ExecuteAutoSync

Description:    Synchronize the buffer using auto-sync informations. This function doesn't execute anything
                if the auto-sync option is not enabled (see LBuffer_SetOptions).

Parameters:     IN  hBuffer                 Handle identifying the buffer.
                IN  hDevThread              Handle identifying the device thread with which the buffer will be
                                            synchronized. If this parameter is MNULL, the buffer will be
                                            synchronized with the CPU.
                IN bReadOnly                Synchronize only for a read only operation (no write is expected).

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_TIMEOUT             The synchronization is not completed yet because the waiting
                                            duration has exceeded the timeout set within the options (see
                                            LBuffer_SetOptions).
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_ExecuteAutoSync(
        LBuffer_Handle       hBuffer,
        LDeviceThread_Handle hDevThread,
        MBOOL32              bReadOnly);

// Same as LBuffer_ExecuteAutoSync but only for the regions that intersect with the given rectangle (poRect).
LAPI LStatus LBuffer_ExecuteAutoSyncRect(
        LBuffer_Handle          hBuffer,
        LDeviceThread_Handle    hDevThread,
        MBOOL32                 bReadOnly,
        LRECT32*                poRect);

/************************************************************************************************************\

Function:       LBuffer_AddAutoSyncRect

Description:    Add a region into the auto-sync list of the given buffer.

Parameters:     IN  hBuffer                 Handle identifying the buffer.
                IN  hDevThread              Handle identifying the device thread on which the buffer region
                                            will be synchronized.
                IN bReadOnly                Synchronize only for a read only operation (no write is expected).
                IN poRect                   Region rectangle. If MNULL, the region covers the whole buffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_UNSUPPORTED         Function failed because the auto-sync option is not enabled (see
                                            LBuffer_SetOptions).
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBuffer_AddAutoSyncRect(
            LBuffer_Handle          hBuffer,
            LDeviceThread_Handle    hDevThread,
            MBOOL32                 bReadOnly,
            LRECT32*                poRect);

/************************************************************************************************************\

Function:       LBuffer_CheckEngineAccess

Description:    Verifies accesses to be performed to the given buffer using any SV2 engine. It returns an
                HDCP access violation flag if there is a risk that the security of HDCP content is
                compromised.

Parameters:     IN  hSrcBuffer          Source buffer of the SV2 engine operation.
                IN  hDstBuffer          Destination buffer of the SV2 engine operation.
                IN  bCrypt              A cryptographics operation (enc / dec) is performed.
                OUT pbViolation         MTRUE  if there will be a risk of HDCP access violation.
                                        MFALSE if it is guaranteed that there will be no HDCP access 
                                               violation.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.

Comments:       A call to this function will not change the buffer internal states.

\************************************************************************************************************/
LAPI LStatus LBuffer_CheckEngineAccess(
    LBuffer_Handle hSrcBuffer,
    LBuffer_Handle hDstBuffer,
    MBOOL32        bCrypt,
    MBOOL32*       pbViolation);

/************************************************************************************************************\

Function:       LBuffer_CheckDirectAccess

Description:    Verifies accesses to be performed to the given buffer using BeginAccess or BeginBusAccess. It
                returns an HDCP access violation flag if there is a risk that the security of HDCP content is
                compromised.

Parameters:     IN  hBuffer             Buffer where a BeginAccess or BeginBusAccess will be requested.
                OUT pbViolation         MTRUE  if there will be a risk of HDCP access violation.
                                        MFALSE if it is guaranteed that there will be no HDCP access 
                                               violation.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.

Comments:       A call to this function will not change the buffer internal states.

\************************************************************************************************************/
LAPI LStatus LBuffer_CheckDirectAccess(
    LBuffer_Handle hBuffer,
    MBOOL32*       pbViolation);

// -----------------------------------------------------------------------------------------------------------
//                       I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------
                
// Generic Helper macros (LBuffer).
#define Declare_LBuffer_LinearAttributes( oName ) \
    LBuffer_LinearAttributes                oName = { LBuffer_Type_LINEAR }
#define Declare_LBuffer_VideoAttributes( oName ) \
    LBuffer_VideoAttributes                 oName = { LBuffer_Type_VIDEO }
#define Declare_LBuffer_SystemLinearAttributes( oName ) \
    LBuffer_SystemLinearAttributes          oName = { LBuffer_Type_SYSTEM_LINEAR }
#define Declare_LBuffer_SystemVideoAttributes( oName ) \
    LBuffer_SystemVideoAttributes           oName = { LBuffer_Type_SYSTEM_VIDEO }
#define Declare_LBuffer_ExternalLinearAttributes( oName ) \
    LBuffer_ExternalLinearAttributes        oName = { LBuffer_Type_EXTERNAL_LINEAR }
#define Declare_LBuffer_ExternalVideoAttributes( oName ) \
    LBuffer_ExternalVideoAttributes         oName = { LBuffer_Type_EXTERNAL_VIDEO }
#define Declare_LBuffer_BusDescriptorLinearAttributes( oName ) \
    LBuffer_BusDescriptorLinearAttributes   oName = { LBuffer_Type_FROM_BUS_DESCRIPTOR_LINEAR }
#define Declare_LBuffer_BusDescriptorVideoAttributes( oName ) \
    LBuffer_BusDescriptorVideoAttributes    oName = { LBuffer_Type_FROM_BUS_DESCRIPTOR_VIDEO }
#define Declare_LBuffer_LUID( oName ) \
    LBuffer_LUID                            oName = { LBuffer_Type_FROM_LUID }

#if defined (__cplusplus)
}
#endif


#endif  // #ifndef INC_IBUFFER_H
